<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\WaSent;
use App\Models\WaReceived;
use App\Models\WaAccount;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AdminController extends Controller
{
    /**
     * Admin dashboard
     */
    public function dashboard()
    {
        $stats = [
            'total_users' => User::count(),
            'active_users' => User::where('suspended', 0)->count(),
            'total_wa_accounts' => WaAccount::where('status', 1)->count(),
            'messages_today' => WaSent::whereDate('create_date', today())->count(),
            'received_today' => WaReceived::whereDate('receive_date', today())->count(),
            'revenue_this_month' => Transaction::whereMonth('create_date', now()->month)
                ->sum('price'),
        ];

        // Get recent users
        $recentUsers = User::orderBy('create_date', 'desc')
            ->limit(5)
            ->get();

        // Get chart data
        $chartData = $this->getAdminChartData(7);

        return view('admin.dashboard', compact('stats', 'recentUsers', 'chartData'));
    }

    /**
     * List all users
     */
    public function users(Request $request)
    {
        $query = User::with('subscription.package');

        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        if ($request->has('status')) {
            $query->where('suspended', $request->status == 'suspended' ? 1 : 0);
        }

        $users = $query->orderBy('create_date', 'desc')
            ->paginate(25);

        return view('admin.users.index', compact('users'));
    }

    /**
     * Edit user
     */
    public function editUser(User $user)
    {
        return view('admin.users.edit', compact('user'));
    }

    /**
     * Update user
     */
    public function updateUser(Request $request, User $user)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'role' => 'required|integer',
            'credits' => 'required|numeric',
            'suspended' => 'required|boolean',
        ]);

        $user->update($validated);

        return redirect()->route('admin.users')
            ->with('success', __('admin.user_updated'));
    }

    /**
     * Suspend/Unsuspend user
     */
    public function toggleUserStatus(User $user)
    {
        $user->update([
            'suspended' => !$user->suspended,
        ]);

        return back()->with('success', __('admin.user_status_updated'));
    }

    /**
     * Delete user
     */
    public function deleteUser(User $user)
    {
        // Don't allow deleting the main admin
        if ($user->id === 1) {
            return back()->with('error', __('admin.cannot_delete_admin'));
        }

        $user->delete();

        return redirect()->route('admin.users')
            ->with('success', __('admin.user_deleted'));
    }

    /**
     * Get chart data for admin dashboard
     */
    private function getAdminChartData(int $days): array
    {
        $labels = [];
        $usersData = [];
        $messagesData = [];

        for ($i = $days - 1; $i >= 0; $i--) {
            $date = now()->subDays($i)->format('Y-m-d');
            $labels[] = now()->subDays($i)->format('M d');

            $usersData[] = User::whereDate('create_date', $date)->count();
            $messagesData[] = WaSent::whereDate('create_date', $date)->count();
        }

        return [
            'labels' => $labels,
            'users' => $usersData,
            'messages' => $messagesData,
        ];
    }
}
