<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Language;

class LanguageController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'admin']);
    }

    public function index()
    {
        $languages = Language::all();
        $translations = $this->getDefaultTranslations();
        return view('admin.languages.index', compact('languages', 'translations'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:5|unique:languages'
        ]);

        $language = Language::create([
            'name' => $request->name,
            'code' => $request->code,
            'rtl' => $request->has('rtl') ? 1 : 0,
            'is_default' => $request->has('is_default') ? 1 : 0
        ]);

        if ($request->has('is_default')) {
            Language::where('id', '!=', $language->id)->update(['is_default' => 0]);
        }

        return redirect()->route('admin.languages.index')->with('success', __('language_created'));
    }

    public function update(Request $request, $id)
    {
        $language = Language::findOrFail($id);

        $language->update([
            'name' => $request->name,
            'code' => $request->code,
            'rtl' => $request->has('rtl') ? 1 : 0,
            'is_default' => $request->has('is_default') ? 1 : 0
        ]);

        if ($request->has('is_default')) {
            Language::where('id', '!=', $language->id)->update(['is_default' => 0]);
        }

        return redirect()->route('admin.languages.index')->with('success', __('language_updated'));
    }

    public function destroy($id)
    {
        $language = Language::findOrFail($id);

        if ($language->is_default) {
            return back()->with('error', __('cannot_delete_default_language'));
        }

        $language->delete();
        return redirect()->route('admin.languages.index')->with('success', __('language_deleted'));
    }

    public function translate(Request $request, $id)
    {
        $language = Language::findOrFail($id);

        $language->update([
            'translations' => json_encode($request->translations ?? [])
        ]);

        return redirect()->route('admin.languages.index')->with('success', __('translations_saved'));
    }

    protected function getDefaultTranslations()
    {
        return [
            'dashboard' => 'Dashboard',
            'settings' => 'Settings',
            'users' => 'Users',
            'messages' => 'Messages',
            'contacts' => 'Contacts',
            // ... more translations
        ];
    }
}
