<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Package;

class PackageController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'admin']);
    }

    public function index()
    {
        $packages = Package::orderBy('price', 'asc')->get();
        return view('admin.packages.index', compact('packages'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'price' => 'required|numeric|min:0',
            'duration' => 'required|integer|min:1'
        ]);

        Package::create([
            'name' => $request->name,
            'price' => $request->price,
            'duration' => $request->duration,
            'send_limit' => $request->send_limit ?? 0,
            'receive_limit' => $request->receive_limit ?? 0,
            'device_limit' => $request->device_limit ?? 0,
            'contact_limit' => $request->contact_limit ?? 0,
            'wa_account_limit' => $request->wa_account_limit ?? 0,
            'services' => is_array($request->services) ? implode(',', $request->services) : '',
            'description' => $request->description,
            'status' => $request->has('status') ? 1 : 0,
            'hidden' => $request->has('hidden') ? 1 : 0
        ]);

        return redirect()->route('admin.packages.index')->with('success', __('package_created'));
    }

    public function update(Request $request, $id)
    {
        $package = Package::findOrFail($id);

        $request->validate([
            'name' => 'required|string|max:255',
            'price' => 'required|numeric|min:0',
            'duration' => 'required|integer|min:1'
        ]);

        $package->update([
            'name' => $request->name,
            'price' => $request->price,
            'duration' => $request->duration,
            'send_limit' => $request->send_limit ?? 0,
            'receive_limit' => $request->receive_limit ?? 0,
            'device_limit' => $request->device_limit ?? 0,
            'contact_limit' => $request->contact_limit ?? 0,
            'wa_account_limit' => $request->wa_account_limit ?? 0,
            'services' => is_array($request->services) ? implode(',', $request->services) : '',
            'description' => $request->description,
            'status' => $request->has('status') ? 1 : 0,
            'hidden' => $request->has('hidden') ? 1 : 0
        ]);

        return redirect()->route('admin.packages.index')->with('success', __('package_updated'));
    }

    public function destroy($id)
    {
        Package::findOrFail($id)->delete();
        return redirect()->route('admin.packages.index')->with('success', __('package_deleted'));
    }
}
