<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Plugin;
use Illuminate\Support\Facades\Storage;
use ZipArchive;

class PluginController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'admin']);
    }

    public function index()
    {
        $plugins = Plugin::all();
        return view('admin.plugins.index', compact('plugins'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'plugin_file' => 'required|file|mimes:zip|max:50000'
        ]);

        $file = $request->file('plugin_file');
        $zipPath = $file->store('temp');

        $zip = new ZipArchive();
        if ($zip->open(Storage::path($zipPath)) === true) {
            // Read plugin.json from zip
            $pluginJson = $zip->getFromName('plugin.json');
            if (!$pluginJson) {
                Storage::delete($zipPath);
                return back()->with('error', __('invalid_plugin_package'));
            }

            $pluginData = json_decode($pluginJson, true);

            // Extract to plugins directory
            $extractPath = base_path('plugins/' . $pluginData['slug']);
            $zip->extractTo($extractPath);
            $zip->close();

            // Create plugin record
            Plugin::create([
                'name' => $pluginData['name'],
                'slug' => $pluginData['slug'],
                'version' => $pluginData['version'],
                'description' => $pluginData['description'] ?? '',
                'author' => $pluginData['author'] ?? '',
                'status' => 0
            ]);

            Storage::delete($zipPath);
            return redirect()->route('admin.plugins.index')->with('success', __('plugin_installed'));
        }

        Storage::delete($zipPath);
        return back()->with('error', __('failed_to_extract_plugin'));
    }

    public function update(Request $request, $id)
    {
        $plugin = Plugin::findOrFail($id);
        $plugin->update(['status' => $request->has('status') ? 1 : 0]);

        return redirect()->route('admin.plugins.index')->with('success', __('plugin_updated'));
    }

    public function destroy($id)
    {
        $plugin = Plugin::findOrFail($id);

        // Delete plugin files
        $pluginPath = base_path('plugins/' . $plugin->slug);
        if (is_dir($pluginPath)) {
            $this->deleteDirectory($pluginPath);
        }

        $plugin->delete();
        return redirect()->route('admin.plugins.index')->with('success', __('plugin_deleted'));
    }

    protected function deleteDirectory($dir)
    {
        if (!is_dir($dir)) return;
        
        $objects = scandir($dir);
        foreach ($objects as $object) {
            if ($object != "." && $object != "..") {
                if (is_dir($dir . "/" . $object)) {
                    $this->deleteDirectory($dir . "/" . $object);
                } else {
                    unlink($dir . "/" . $object);
                }
            }
        }
        rmdir($dir);
    }
}
