<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\WaServer;

class WaServerController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'admin']);
    }

    public function index()
    {
        $waservers = WaServer::withCount('accounts')->get();
        return view('admin.waservers.index', compact('waservers'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'url' => 'required|url',
            'port' => 'required|integer',
            'secret' => 'required|string'
        ]);

        WaServer::create([
            'name' => $request->name,
            'url' => $request->url,
            'port' => $request->port,
            'secret' => $request->secret,
            'max_accounts' => $request->max_accounts ?? 100,
            'status' => $request->has('status') ? 1 : 0
        ]);

        return redirect()->route('admin.waservers.index')->with('success', __('waserver_created'));
    }

    public function update(Request $request, $id)
    {
        $server = WaServer::findOrFail($id);

        $server->update([
            'name' => $request->name,
            'url' => $request->url,
            'port' => $request->port,
            'secret' => $request->secret,
            'max_accounts' => $request->max_accounts ?? 100,
            'status' => $request->has('status') ? 1 : 0
        ]);

        return redirect()->route('admin.waservers.index')->with('success', __('waserver_updated'));
    }

    public function destroy($id)
    {
        WaServer::findOrFail($id)->delete();
        return redirect()->route('admin.waservers.index')->with('success', __('waserver_deleted'));
    }

    public function test($id)
    {
        $server = WaServer::findOrFail($id);
        
        try {
            $response = \Http::timeout(5)->get($server->url . ':' . $server->port . '/health');
            
            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => __('connection_success')]);
            }
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()]);
        }

        return response()->json(['success' => false, 'message' => __('connection_failed')]);
    }
}
