<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Device;
use App\Models\Sent;
use App\Models\Received;
use App\Models\Ussd;
use App\Models\Quota;
use App\Models\Subscription;
use App\Models\Package;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

/**
 * Android Device Controller
 * يتعامل مع طلبات تطبيق Android للـ SMS
 */
class AndroidController extends Controller
{
    /**
     * جلب الرسائل المعلقة للإرسال عبر Android
     * GET /remote/messages?device=&hash=
     */
    public function messages(Request $request)
    {
        $deviceId = $request->input('device');
        $hash = $request->input('hash');
        $diff = min(max((int)$request->input('diff', 20), 1), 50);

        if (!$deviceId || !$hash) {
            return response()->json(['status' => 500]);
        }

        // التحقق من الجهاز
        $device = Device::where('did', $deviceId)->first();
        if (!$device) {
            return response()->json(['status' => 500]);
        }

        // التحقق من hash
        $expectedHash = md5($device->uid . $deviceId);
        if ($hash !== $expectedHash) {
            return response()->json(['status' => 500]);
        }

        // التحقق من المستخدم
        $user = User::find($device->uid);
        if (!$user || $user->suspended) {
            return response()->json(['status' => 500]);
        }

        // التحقق من الاشتراك
        $subscription = $this->getSubscription($device->uid);
        if (!$subscription) {
            return response()->json(['status' => 500]);
        }

        // جلب الرسائل المعلقة
        $messages = Sent::where('uid', $device->uid)
            ->where('did', $deviceId)
            ->where('status', 1) // معلق
            ->where('gateway', 0) // جهاز Android
            ->orderBy('priority', 'asc')
            ->orderBy('id', 'asc')
            ->limit($diff)
            ->get();

        if ($messages->isEmpty()) {
            return response()->json(['status' => 200, 'data' => []]);
        }

        // التحقق من حدود الإرسال
        $quota = $this->getOrCreateQuota($device->uid);
        $messageContainer = [];

        foreach ($messages as $message) {
            // التحقق من الحد
            if ($quota->sent >= $subscription['send_limit'] && $subscription['send_limit'] > 0) {
                $message->update(['status' => 4]); // فشل بسبب الحد
                continue;
            }

            $messageContainer[] = [
                'id' => $message->id,
                'phone' => $message->phone,
                'message' => $message->message,
                'sim' => $message->sim,
            ];

            // تحديث الحالة
            $message->update(['status' => 2]);

            // زيادة العداد
            $quota->increment('sent');
        }

        return response()->json([
            'status' => 200,
            'data' => [
                'random_send' => $device->random_send,
                'random_min' => $device->random_min,
                'random_max' => $device->random_max,
                'messages' => $messageContainer
            ]
        ]);
    }

    /**
     * تحديث حالة الرسالة
     * POST /remote/status
     */
    public function status(Request $request)
    {
        $messageId = $request->input('id');
        $status = $request->input('status');
        $statusCode = $request->input('status_code');

        if (!$messageId || !$status) {
            return response()->json(['status' => 500]);
        }

        $message = Sent::find($messageId);
        if (!$message) {
            return response()->json(['status' => 500]);
        }

        $updateData = ['status' => $status];
        if ($statusCode) {
            $updateData['status_code'] = $statusCode;
        }

        $message->update($updateData);

        return response()->json(['status' => 200]);
    }

    /**
     * استقبال رسالة SMS واردة
     * POST /remote/received
     */
    public function received(Request $request)
    {
        $deviceId = $request->input('device');
        $hash = $request->input('hash');
        $phone = $request->input('phone');
        $message = $request->input('message');
        $sim = $request->input('sim', 1);

        if (!$deviceId || !$hash || !$phone || !$message) {
            return response()->json(['status' => 500]);
        }

        // التحقق من الجهاز
        $device = Device::where('did', $deviceId)->first();
        if (!$device) {
            return response()->json(['status' => 500]);
        }

        // التحقق من hash
        $expectedHash = md5($device->uid . $deviceId);
        if ($hash !== $expectedHash) {
            return response()->json(['status' => 500]);
        }

        // التحقق من إعداد استقبال الرسائل
        if (!$device->receive_sms) {
            return response()->json(['status' => 200]);
        }

        // التحقق من الاشتراك
        $subscription = $this->getSubscription($device->uid);
        if (!$subscription) {
            return response()->json(['status' => 403]);
        }

        // التحقق من حد الاستقبال
        $quota = $this->getOrCreateQuota($device->uid);
        if ($quota->received >= $subscription['receive_limit'] && $subscription['receive_limit'] > 0) {
            return response()->json(['status' => 403]);
        }

        // إنشاء السجل
        $received = Received::create([
            'uid' => $device->uid,
            'rid' => uniqid('sms_'),
            'did' => $deviceId,
            'sim' => $sim,
            'phone' => $phone,
            'message' => $message,
        ]);

        if ($received) {
            $quota->increment('received');
            
            Log::info("SMS received", [
                'id' => $received->id,
                'uid' => $device->uid,
                'phone' => $phone
            ]);
        }

        return response()->json(['status' => 200]);
    }

    /**
     * تحديث نتيجة USSD
     * POST /remote/ussd
     */
    public function ussd(Request $request)
    {
        $ussdId = $request->input('id');
        $response = $request->input('response');
        $status = $request->input('status');

        if (!$ussdId) {
            return response()->json(['status' => 500]);
        }

        $ussd = Ussd::find($ussdId);
        if (!$ussd) {
            return response()->json(['status' => 500]);
        }

        $ussd->update([
            'response' => $response,
            'status' => $status ?? 3, // مكتمل
        ]);

        return response()->json(['status' => 200]);
    }

    /**
     * الحصول على اشتراك المستخدم
     */
    protected function getSubscription(int $uid): ?array
    {
        $subscription = Subscription::where('uid', $uid)
            ->where('expiry_date', '>', now())
            ->first();

        if (!$subscription) {
            $defaultPackage = Package::where('is_default', 1)->first();
            if ($defaultPackage) {
                return $defaultPackage->toArray();
            }
            return null;
        }

        $package = Package::find($subscription->pid);
        return $package ? $package->toArray() : null;
    }

    /**
     * إنشاء أو جلب quota المستخدم
     */
    protected function getOrCreateQuota(int $uid): Quota
    {
        return Quota::firstOrCreate(
            ['uid' => $uid],
            [
                'sent' => 0,
                'received' => 0,
                'wa_sent' => 0,
                'wa_received' => 0,
                'ussd' => 0
            ]
        );
    }
}
