<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Key;
use App\Models\Contact;
use App\Models\WaSent;
use App\Models\WaAccount;
use App\Services\WhatsappBridgeService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ApiController extends Controller
{
    protected WhatsappBridgeService $bridgeService;

    public function __construct(WhatsappBridgeService $bridgeService)
    {
        $this->bridgeService = $bridgeService;
    }

    /**
     * Authenticate API request and get user
     */
    private function authenticate(Request $request): ?array
    {
        $apiKey = $request->header('X-Api-Key') ?? $request->input('api_key');
        
        if (!$apiKey) {
            return null;
        }

        $key = Key::where('secret', $apiKey)->first();
        
        if (!$key) {
            return null;
        }

        $user = User::find($key->uid);
        
        if (!$user || $user->suspended) {
            return null;
        }

        return [
            'user' => $user,
            'key' => $key,
        ];
    }

    /**
     * Get account info
     */
    public function getAccount(Request $request)
    {
        $auth = $this->authenticate($request);
        
        if (!$auth) {
            return response()->json([
                'status' => 401,
                'message' => 'Unauthorized',
            ], 401);
        }

        $user = $auth['user'];

        return response()->json([
            'status' => 200,
            'data' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'credits' => $user->credits,
                'subscription' => $user->subscription ? [
                    'package' => $user->subscription->package->name ?? null,
                    'expires' => $user->subscription->date,
                ] : null,
            ],
        ]);
    }

    /**
     * Send WhatsApp message
     */
    public function sendWhatsapp(Request $request)
    {
        $auth = $this->authenticate($request);
        
        if (!$auth) {
            return response()->json([
                'status' => 401,
                'message' => 'Unauthorized',
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'account' => 'required|string',
            'recipient' => 'required|string',
            'message' => 'required|string',
            'type' => 'nullable|string|in:text,image,document,audio,video',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 400,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 400);
        }

        $user = $auth['user'];
        
        // Find the WhatsApp account
        $account = WaAccount::where('uid', $user->id)
            ->where(function($q) use ($request) {
                $q->where('unique', $request->account)
                  ->orWhere('id', $request->account)
                  ->orWhere('phone', $request->account);
            })
            ->first();

        if (!$account) {
            return response()->json([
                'status' => 404,
                'message' => 'WhatsApp account not found',
            ], 404);
        }

        if (!$account->isConnected()) {
            return response()->json([
                'status' => 400,
                'message' => 'WhatsApp account is not connected',
            ], 400);
        }

        // Create message record
        $message = WaSent::create([
            'uid' => $user->id,
            'wid' => $account->wid,
            'unique' => $account->unique,
            'phone' => $this->formatPhone($request->recipient),
            'message' => $request->message,
            'message_type' => $this->getMessageType($request->type ?? 'text'),
            'status' => 1, // Pending
            'api' => $auth['key']->secret,
        ]);

        // Send via bridge
        $success = $this->bridgeService->sendMessage(
            $account,
            $message->phone,
            $request->message,
            ['id' => $message->id]
        );

        if ($success) {
            $message->update(['status' => 2]); // Sent
            return response()->json([
                'status' => 200,
                'message' => 'Message sent successfully',
                'data' => [
                    'id' => $message->id,
                    'phone' => $message->phone,
                    'status' => 'sent',
                ],
            ]);
        }

        $message->update(['status' => 5]); // Failed
        return response()->json([
            'status' => 500,
            'message' => 'Failed to send message',
            'data' => [
                'id' => $message->id,
                'status' => 'failed',
            ],
        ], 500);
    }

    /**
     * Get contacts
     */
    public function getContacts(Request $request)
    {
        $auth = $this->authenticate($request);
        
        if (!$auth) {
            return response()->json([
                'status' => 401,
                'message' => 'Unauthorized',
            ], 401);
        }

        $contacts = Contact::where('uid', $auth['user']->id)
            ->orderBy('create_date', 'desc')
            ->limit($request->input('limit', 100))
            ->offset($request->input('offset', 0))
            ->get(['id', 'name', 'phone', 'groups', 'create_date']);

        return response()->json([
            'status' => 200,
            'data' => $contacts,
        ]);
    }

    /**
     * Add contact
     */
    public function addContact(Request $request)
    {
        $auth = $this->authenticate($request);
        
        if (!$auth) {
            return response()->json([
                'status' => 401,
                'message' => 'Unauthorized',
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:50',
            'group' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 400,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 400);
        }

        $contact = Contact::create([
            'uid' => $auth['user']->id,
            'name' => $request->name,
            'phone' => $this->formatPhone($request->phone),
            'groups' => $request->group ?? '',
        ]);

        return response()->json([
            'status' => 200,
            'message' => 'Contact added successfully',
            'data' => $contact,
        ]);
    }

    /**
     * Get WhatsApp accounts
     */
    public function getWhatsappAccounts(Request $request)
    {
        $auth = $this->authenticate($request);
        
        if (!$auth) {
            return response()->json([
                'status' => 401,
                'message' => 'Unauthorized',
            ], 401);
        }

        $accounts = WaAccount::where('uid', $auth['user']->id)
            ->get(['id', 'unique', 'phone', 'name', 'status', 'create_date']);

        return response()->json([
            'status' => 200,
            'data' => $accounts,
        ]);
    }

    /**
     * Format phone number
     */
    private function formatPhone(string $phone): string
    {
        $phone = preg_replace('/[^0-9+]/', '', $phone);
        
        if (!str_starts_with($phone, '+')) {
            $phone = '+' . $phone;
        }
        
        return $phone;
    }

    /**
     * Get message type ID
     */
    private function getMessageType(string $type): int
    {
        return match($type) {
            'text' => 1,
            'image' => 2,
            'video' => 3,
            'audio' => 4,
            'document' => 5,
            'location' => 6,
            'contact' => 7,
            'sticker' => 8,
            default => 1,
        };
    }
}
