<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Subscription;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rules\Password;

class AuthController extends Controller
{
    /**
     * Show login form
     */
    public function showLogin()
    {
        if (Auth::check()) {
            return redirect()->route('dashboard');
        }
        return view('auth.login');
    }

    /**
     * Handle login request
     */
    public function login(Request $request)
    {
        $credentials = $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        if (Auth::attempt($credentials, $request->boolean('remember'))) {
            $user = Auth::user();
            
            if ($user->suspended) {
                Auth::logout();
                return back()->withErrors([
                    'email' => __('auth.suspended'),
                ]);
            }

            $request->session()->regenerate();
            
            // Redirect admin to admin panel
            if ($user->isAdmin()) {
                return redirect()->intended(route('admin.dashboard'));
            }

            return redirect()->intended(route('dashboard'));
        }

        return back()->withErrors([
            'email' => __('auth.failed'),
        ])->onlyInput('email');
    }

    /**
     * Show registration form
     */
    public function showRegister()
    {
        if (!Setting::get('registration_enabled', true)) {
            return redirect()->route('login')
                ->with('error', __('auth.registration_disabled'));
        }

        if (Auth::check()) {
            return redirect()->route('dashboard');
        }

        return view('auth.register');
    }

    /**
     * Handle registration request
     */
    public function register(Request $request)
    {
        if (!Setting::get('registration_enabled', true)) {
            return redirect()->route('login')
                ->with('error', __('auth.registration_disabled'));
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => ['required', 'confirmed', Password::min(8)],
            'country' => 'required|string|size:2',
        ]);

        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'country' => strtoupper($validated['country']),
            'role' => 2, // Regular user
            'confirmed' => Setting::get('email_verification', true) ? 0 : 1,
        ]);

        // Create default subscription
        $defaultPackage = Setting::get('default_package', 1);
        Subscription::create([
            'uid' => $user->id,
            'pid' => $defaultPackage,
            'tid' => 0,
            'date' => now()->addDays(30),
        ]);

        if (Setting::get('email_verification', true)) {
            // TODO: Send verification email
            return redirect()->route('login')
                ->with('success', __('auth.verify_email'));
        }

        Auth::login($user);
        return redirect()->route('dashboard');
    }

    /**
     * Handle logout request
     */
    public function logout(Request $request)
    {
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        return redirect()->route('login');
    }

    /**
     * Show forgot password form
     */
    public function showForgotPassword()
    {
        return view('auth.forgot-password');
    }

    /**
     * Handle forgot password request
     */
    public function forgotPassword(Request $request)
    {
        $request->validate([
            'email' => 'required|email|exists:users,email',
        ]);

        // TODO: Send password reset email

        return back()->with('success', __('auth.reset_link_sent'));
    }
}
