<?php

namespace App\Http\Controllers\Dashboard\Hosts;

use App\Http\Controllers\Controller;
use App\Models\WaAccount;
use App\Services\WhatsappBridgeService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class WaAccountController extends Controller
{
    protected WhatsappBridgeService $bridgeService;

    public function __construct(WhatsappBridgeService $bridgeService)
    {
        $this->bridgeService = $bridgeService;
    }

    public function index(Request $request)
    {
        $accounts = WaAccount::where('uid', Auth::id())
            ->orderBy('create_date', 'desc')
            ->paginate(25);
            
        return view('dashboard.hosts.whatsapp', compact('accounts'));
    }

    public function create()
    {
        $server = $this->bridgeService->getServer();
        
        if (!$server) {
            return back()->with('error', __('messages.no_whatsapp_server'));
        }

        $user = Auth::user();
        $hash = md5($user->id . time());
        
        $account = WaAccount::create([
            'uid' => $user->id,
            'unique' => uniqid(time() . $hash),
            'status' => 0,
        ]);

        $qr = $this->bridgeService->create(
            $server->secret,
            $user->id,
            $hash,
            $account->id,
            $server->address,
            $server->port,
            $account->unique
        );

        if (!$qr) {
            $account->delete();
            return back()->with('error', __('messages.qr_generation_failed'));
        }

        return view('dashboard.whatsapp.scan', compact('account', 'qr'));
    }

    public function checkStatus(WaAccount $account)
    {
        $status = $this->bridgeService->getAccountStatus($account);

        if ($status && isset($status->connected) && $status->connected) {
            $account->update([
                'status' => 1,
                'phone' => $status->phone ?? null,
                'name' => $status->name ?? null,
            ]);

            return response()->json([
                'success' => true,
                'connected' => true,
                'phone' => $status->phone,
                'name' => $status->name,
            ]);
        }

        return response()->json(['success' => true, 'connected' => false]);
    }

    public function update(Request $request, WaAccount $account)
    {
        $account->update($request->only(['name', 'receive_chats', 'random_send', 'random_min', 'random_max']));
        return back()->with('success', __('messages.account_updated'));
    }

    public function destroy(WaAccount $account)
    {
        $server = $this->bridgeService->getServer();
        
        if ($server && $account->unique) {
            $this->bridgeService->delete(
                $server->secret,
                $server->address,
                $server->port,
                $account->unique
            );
        }

        $account->delete();
        return back()->with('success', __('messages.account_deleted'));
    }
}
