<?php

namespace App\Http\Controllers\Dashboard;

use App\Http\Controllers\Controller;
use App\Models\WaGroup;
use App\Models\WaAccount;
use App\Services\WhatsappBridgeService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class WaGroupController extends Controller
{
    protected WhatsappBridgeService $bridgeService;

    public function __construct(WhatsappBridgeService $bridgeService)
    {
        $this->bridgeService = $bridgeService;
    }

    public function index(Request $request)
    {
        $groups = WaGroup::where('uid', Auth::id())
            ->with('account')
            ->orderBy('create_date', 'desc')
            ->paginate(25);
            
        $accounts = WaAccount::where('uid', Auth::id())->where('status', 1)->get();
        
        return view('dashboard.whatsapp.groups', compact('groups', 'accounts'));
    }

    public function sync(Request $request)
    {
        $request->validate(['account_id' => 'required|integer']);
        
        $account = WaAccount::findOrFail($request->account_id);
        $server = $this->bridgeService->getServer();
        
        if (!$server) {
            return back()->with('error', __('messages.no_whatsapp_server'));
        }

        $groups = $this->bridgeService->getGroups(
            $server->secret,
            $server->address,
            $server->port,
            $account->unique
        );

        if ($groups) {
            // Clear old groups
            WaGroup::where('waid', $account->id)->delete();
            
            // Insert new groups
            foreach ($groups as $group) {
                WaGroup::create([
                    'uid' => Auth::id(),
                    'waid' => $account->id,
                    'jid' => $group['jid'] ?? $group['JID'],
                    'name' => $group['name'] ?? $group['Name'],
                    'members' => $group['members'] ?? 0,
                ]);
            }
            
            return back()->with('success', __('messages.groups_synced', ['count' => count($groups)]));
        }

        return back()->with('error', __('messages.sync_failed'));
    }

    public function members(WaGroup $group)
    {
        $account = $group->account;
        $server = $this->bridgeService->getServer();
        
        if (!$server) {
            return response()->json(['error' => 'No server'], 400);
        }

        $members = $this->bridgeService->getGroupMembers(
            $server->secret,
            $server->address,
            $server->port,
            $account->unique,
            $group->jid
        );

        return response()->json($members ?: []);
    }
}
