<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;

/**
 * Legacy Data Seeder
 * يستورد البيانات من ملف SQL القديم إلى قاعدة بيانات Laravel
 * 
 * ملاحظة: يجب تشغيل migrations أولاً قبل تشغيل هذا الـ Seeder
 */
class LegacyDataSeeder extends Seeder
{
    /**
     * تشغيل الـ Seeder
     */
    public function run(): void
    {
        $sqlFile = '/app/new_archive/flywhats_database.sql';

        if (!File::exists($sqlFile)) {
            $this->command->error("SQL file not found at: {$sqlFile}");
            return;
        }

        $this->command->info('Starting legacy data import...');
        
        // قراءة ملف SQL
        $sql = File::get($sqlFile);
        
        // استخراج عبارات INSERT
        preg_match_all('/INSERT INTO `(\w+)` (.*?);/s', $sql, $matches, PREG_SET_ORDER);

        $this->command->info('Found ' . count($matches) . ' INSERT statements');

        $importedTables = [];
        $errors = [];

        foreach ($matches as $match) {
            $tableName = $match[1];
            $insertStatement = "INSERT INTO `{$tableName}` {$match[2]};";
            
            try {
                DB::unprepared($insertStatement);
                
                if (!isset($importedTables[$tableName])) {
                    $importedTables[$tableName] = 0;
                }
                $importedTables[$tableName]++;
                
            } catch (\Exception $e) {
                $errors[] = "Error importing {$tableName}: " . $e->getMessage();
            }
        }

        // عرض النتائج
        $this->command->info('');
        $this->command->info('=== Import Summary ===');
        
        foreach ($importedTables as $table => $count) {
            $this->command->line("  - {$table}: {$count} statement(s)");
        }

        if (!empty($errors)) {
            $this->command->warn('');
            $this->command->warn('=== Errors ===');
            foreach (array_unique($errors) as $error) {
                $this->command->error($error);
            }
        }

        $this->command->info('');
        $this->command->info('Legacy data import completed!');
    }

    /**
     * استيراد جدول محدد فقط
     */
    public function importTable(string $tableName): void
    {
        $sqlFile = '/app/new_archive/flywhats_database.sql';
        
        if (!File::exists($sqlFile)) {
            $this->command->error("SQL file not found");
            return;
        }

        $sql = File::get($sqlFile);
        
        preg_match_all("/INSERT INTO `{$tableName}` (.*?);/s", $sql, $matches, PREG_SET_ORDER);

        if (empty($matches)) {
            $this->command->warn("No INSERT statements found for table: {$tableName}");
            return;
        }

        $this->command->info("Importing {$tableName}...");

        foreach ($matches as $match) {
            $insertStatement = "INSERT INTO `{$tableName}` {$match[1]};";
            
            try {
                DB::unprepared($insertStatement);
            } catch (\Exception $e) {
                $this->command->error($e->getMessage());
            }
        }

        $this->command->info("Done importing {$tableName}");
    }
}
