@extends('layouts.app')

@section('title', __('Messenger'))
@section('page-title', __('WhatsApp Messenger'))

@push('styles')
<style>
    .messenger-container {
        display: grid;
        grid-template-columns: 350px 1fr;
        height: calc(100vh - 180px);
        background: #fff;
        border-radius: 10px;
        overflow: hidden;
        box-shadow: 0 1px 3px rgba(0,0,0,0.1);
    }
    .conversations-list {
        border-right: 1px solid #e2e8f0;
        overflow-y: auto;
    }
    .conversation-item {
        padding: 15px;
        border-bottom: 1px solid #f1f5f9;
        cursor: pointer;
        transition: background 0.2s;
    }
    .conversation-item:hover, .conversation-item.active {
        background: #f8fafc;
    }
    .chat-area {
        display: flex;
        flex-direction: column;
    }
    .chat-header {
        padding: 15px 20px;
        border-bottom: 1px solid #e2e8f0;
        background: #f8fafc;
    }
    .messages-container {
        flex: 1;
        overflow-y: auto;
        padding: 20px;
        background: #f1f5f9;
    }
    .message {
        max-width: 70%;
        margin-bottom: 15px;
        padding: 10px 15px;
        border-radius: 15px;
    }
    .message.sent {
        background: #10b981;
        color: #fff;
        margin-left: auto;
        border-bottom-right-radius: 5px;
    }
    .message.received {
        background: #fff;
        border-bottom-left-radius: 5px;
    }
    .chat-input {
        padding: 15px 20px;
        border-top: 1px solid #e2e8f0;
        background: #fff;
    }
</style>
@endpush

@section('content')
<div class="messenger-container">
    <!-- Conversations List -->
    <div class="conversations-list">
        <div class="p-3 border-bottom">
            <input type="text" class="form-control" placeholder="{{ __('Search conversations...') }}">
        </div>
        @forelse($conversations as $conv)
        <div class="conversation-item" data-phone="{{ $conv->phone }}">
            <div class="d-flex align-items-center gap-3">
                <div class="rounded-circle bg-success text-white d-flex align-items-center justify-content-center" 
                     style="width: 45px; height: 45px;">
                    <i class="bi bi-person"></i>
                </div>
                <div class="flex-grow-1">
                    <div class="fw-semibold">{{ $conv->wa_name ?? $conv->phone }}</div>
                    <small class="text-muted">{{ $conv->phone }}</small>
                </div>
            </div>
        </div>
        @empty
        <div class="text-center text-muted py-5">
            <i class="bi bi-chat-dots fs-1"></i>
            <p class="mt-2">{{ __('No conversations yet') }}</p>
        </div>
        @endforelse
    </div>

    <!-- Chat Area -->
    <div class="chat-area">
        <div class="chat-header">
            <div id="chat-contact-name">{{ __('Select a conversation') }}</div>
        </div>
        <div class="messages-container" id="messages-container">
            <div class="text-center text-muted py-5">
                <i class="bi bi-chat-left-text fs-1"></i>
                <p class="mt-2">{{ __('Select a conversation to view messages') }}</p>
            </div>
        </div>
        <div class="chat-input">
            <form id="send-form" class="d-flex gap-2">
                <select id="account-select" class="form-control" style="width: 200px;" required>
                    @foreach($accounts as $acc)
                    <option value="{{ $acc->id }}">{{ $acc->phone ?? $acc->unique }}</option>
                    @endforeach
                </select>
                <input type="text" id="message-input" class="form-control" placeholder="{{ __('Type a message...') }}" required>
                <button type="submit" class="btn btn-success">
                    <i class="bi bi-send"></i>
                </button>
            </form>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
let currentPhone = null;

document.querySelectorAll('.conversation-item').forEach(item => {
    item.addEventListener('click', function() {
        document.querySelectorAll('.conversation-item').forEach(i => i.classList.remove('active'));
        this.classList.add('active');
        currentPhone = this.dataset.phone;
        document.getElementById('chat-contact-name').textContent = currentPhone;
        loadMessages();
    });
});

function loadMessages() {
    if (!currentPhone) return;
    const account = document.getElementById('account-select').value;
    
    fetch(`{{ route('whatsapp.conversation') }}?phone=${encodeURIComponent(currentPhone)}&account=${account}`)
        .then(r => r.json())
        .then(messages => {
            const container = document.getElementById('messages-container');
            container.innerHTML = '';
            messages.forEach(msg => {
                const div = document.createElement('div');
                div.className = 'message ' + msg.type;
                div.textContent = msg.message;
                container.appendChild(div);
            });
            container.scrollTop = container.scrollHeight;
        });
}

document.getElementById('send-form').addEventListener('submit', function(e) {
    e.preventDefault();
    if (!currentPhone) return alert('{{ __('Select a conversation first') }}');
    
    const message = document.getElementById('message-input').value;
    const account = document.getElementById('account-select').value;
    
    fetch('{{ route('whatsapp.send') }}', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        },
        body: JSON.stringify({
            account_id: account,
            phone: currentPhone,
            message: message
        })
    })
    .then(r => r.json())
    .then(data => {
        document.getElementById('message-input').value = '';
        loadMessages();
    });
});
</script>
@endpush
